/*******************************************************************************************************
 *                                                                                                     *
 *        **********                                                                                   *
 *       ************                                                                                  *
 *      ***        ***                                                                                 *
 *      ***   +++   ***                                                                                *
 *      ***   + +   ***                                                                                *
 *      ***   +                               CHIPCON CC2420 BASIC RF LIBRARY                          *
 *      ***   + +   ***                             Packet transmission                                *
 *      ***   +++   ***                                                                                *
 *      ***        ***                                                                                 *
 *       ************                                                                                  *
 *        **********                                                                                   *
 *                                                                                                     *
 *******************************************************************************************************
 * This file contains packet transmission function.                                                    *
 *                                                                                                     *
 * More information can be found in basic_rf.h                                                         *
 *******************************************************************************************************
 * Compiler: AVR-GCC                                                                                   *
 * Target platform: CC2420DB, CC2420 + any MCU with very few modifications required                    *
 *******************************************************************************************************
 * Revision history:                                                                                   *
 *  $Log: basic_rf_send_packet.c,v $
 *  Revision 1.4  2004/07/26 11:28:10  mbr
 *  Modified RXFIFO flushing by strobing CC2420_SFLUSHRX
 *
 *  Revision 1.3  2004/03/30 14:59:22  mbr
 *  Release for web
 * 
 *
 *
 *******************************************************************************************************/
#include "include/include.h"
UINT8 spiCnt;
UINT8 i;

//-------------------------------------------------------------------------------------------------------
//  BYTE basicRfSendPacket(BASIC_RF_TX_INFO *pRTI)
//
//  DESCRIPTION:
//		Transmits a packet using the IEEE 802.15.4 MAC data packet format with short addresses. CCA is
//		measured only once before backet transmission (not compliant with 802.15.4 CSMA-CA).
//		The function returns:
//			- When pRTI->ackRequest is FALSE: After the transmission has begun (SFD gone high)
//			- When pRTI->ackRequest is TRUE: After the acknowledgment has been received/declared missing.
//		The acknowledgment is received through the FIFOP interrupt.
//
//  ARGUMENTS:
//      BASIC_RF_TX_INFO *pRTI
//          The transmission structure, which contains all relevant info about the packet.
//
//  RETURN VALUE:
//		BOOL
//			Successful transmission (acknowledgment received)
//-------------------------------------------------------------------------------------------------------
BOOL basicRfSendPacket(BASIC_RF_TX_INFO *pRTI) {
	WORD frameControlField;
    UINT8 packetLength;
    BOOL success;
    BYTE spiStatusByte;
//	printf("Just before start waiting for tranceiver to be idle\r\n");
    // Wait until the transceiver is idle
    while (FIFOP_IS_1 || SFD_IS_1);
//	printf("Just after start waiting for tranceiver to be idle\r\n");	

    // Turn off global interrupts to avoid interference on the SPI interface
    DISABLE_GLOBAL_INT();

	// Flush the TX FIFO just in case...
	FASTSPI_STROBE(CC2420_SFLUSHTX);
//	printf("Turning on Receive SRX\r\n");
    // Turn on RX if necessary
    if (!rfSettings.receiveOn) FASTSPI_STROBE(CC2420_SRXON);
//	printf("Waiting for RSSI to become valid\r\n");
    // Wait for the RSSI value to become valid
    do {
        FASTSPI_UPD_STATUS(spiStatusByte);
    } while (!(spiStatusByte & BM(CC2420_RSSI_VALID)));

//	printf("Valid RSSI\r\n");
// without commenting following code it does not work
/*
    // TX begins after the CCA check has passed
    do {
		FASTSPI_STROBE(CC2420_STXONCCA);
		FASTSPI_UPD_STATUS(spiStatusByte);
		halWait(1);
    } while (!(spiStatusByte & BM(CC2420_TX_ACTIVE)));

*/   
    // Write the packet to the TX FIFO (the FCS is appended automatically when AUTOCRC is enabled)
  
    packetLength = pRTI->length + BASIC_RF_PACKET_OVERHEAD_SIZE;
    frameControlField = pRTI->ackRequest ? BASIC_RF_FCF_ACK : BASIC_RF_FCF_NOACK;

//	printf("PacketLength: %x\r\n",packetLength);
//	printf("frameControlField: %x\r\n",frameControlField);
//	printf("rfSettings.txSeqNumber: %x\r\n",rfSettings.txSeqNumber);
//	printf("rfSettings.panId: %x\r\n",rfSettings.panId);
//	printf("pRTI->destAddr: %x\r\n",pRTI->destAddr);
//	printf("rfSettings.myAddr: %x\r\n",rfSettings.myAddr);
//	printf("pRTI->pPayload[0]: %x\r\n",pRTI->pPayload[0]);
//	printf("pRTI->pPayload[1]: %x\r\n",pRTI->pPayload[1]);
	
    FASTSPI_WRITE_FIFO((BYTE*)&packetLength, 1);               // Packet length
    FASTSPI_WRITE_FIFO((BYTE*)&frameControlField, 2);         // Frame control field
    FASTSPI_WRITE_FIFO((BYTE*)&rfSettings.txSeqNumber, 1);    // Sequence number
    FASTSPI_WRITE_FIFO((BYTE*)&rfSettings.panId, 2);          // Dest. PAN ID
    FASTSPI_WRITE_FIFO((BYTE*)&pRTI->destAddr, 2);            // Dest. address
    FASTSPI_WRITE_FIFO((BYTE*)&rfSettings.myAddr, 2);         // Source address
	FASTSPI_WRITE_FIFO((BYTE*)pRTI->pPayload, pRTI->length);  // Payload

	// Wait for the transmission to begin before exiting (makes sure that this function cannot be called
	// a second time, and thereby cancelling the first transmission (observe the FIFOP + SFD test above).

	FASTSPI_STROBE(CC2420_STXONCCA);


// Wait till transmission begins
	while (!SFD_IS_1);
// Wait till transmission finishes
	while (SFD_IS_1);

	// Turn interrupts back on
	ENABLE_GLOBAL_INT();

    // Wait for the acknowledge to be received, if any
    if (pRTI->ackRequest) {
		rfSettings.ackReceived = FALSE;
//	printf("Waiting for ack...have set ackreceived as false\r\n");
		// Wait for the SFD to go low again
		while (SFD_IS_1);
//	printf("SFD is already zero at this point after TX\r\n");

        // We'll enter RX automatically, so just wait until we can be sure that the ack reception should have finished
        // The timeout consists of a 12-symbol turnaround time, the ack packet duration, and a small margin
        halWait((12 * BASIC_RF_SYMBOL_DURATION) + (BASIC_RF_ACK_DURATION) + (2 * BASIC_RF_SYMBOL_DURATION) + 100);
//	printf("After waiting for ack\r\n");
		// If an acknowledgment has been received (by the FIFOP interrupt), the ackReceived flag should be set
		success = rfSettings.ackReceived;
    } else {
        success= TRUE;
    }

	// Turn off the receiver if it should not continue to be enabled
    DISABLE_GLOBAL_INT();
	if (!rfSettings.receiveOn) FASTSPI_STROBE(CC2420_SRFOFF);
    ENABLE_GLOBAL_INT();

    // Increment the sequence number, and return the result
    rfSettings.txSeqNumber++;
    return success;

} // halRfSendPacket

